<?php

/**
 * class-ng-scroll-box.php
 *
 * @package   ng-scroll-box
 * @copyright Copyright (c) 2015, Ashley Evans
 * @license   GPL2+
 */
class NG_Scroll_Box {

	/**
	 * The single instance of the plugin.
	 *
	 * @since  1.0.0
	 * @access private
	 * @var NG_Scroll_Box
	 */
	private static $_instance = null;

	/**
	 * The unique identifier of this plugin.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      string $plugin_name The string used to uniquely identify this plugin.
	 */
	protected $plugin_name;

	/**
	 * The array of plugin settings.
	 *
	 * @since  1.0.0
	 * @access public
	 * @var array
	 */
	public $settings;

	/**
	 * The main plugin file.
	 *
	 * @since  1.0.0
	 * @access protected
	 * @var string Main plugin file
	 */
	protected $file;

	/**
	 * The current version of the plugin.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      string $version The current version of the plugin.
	 */
	protected $version;

	/**
	 * Constructor
	 *
	 * Sets up all variables, initializes the settings, and
	 * adds our shortcode.
	 *
	 * @param string $file
	 * @param string $version
	 *
	 * @access public
	 * @since  1.0.0
	 * @return void
	 */
	public function __construct( $file = '', $version = '1.0.0' ) {

		$this->plugin_name = 'ng-scroll-box';
		$this->file        = $file;
		$this->version     = $version;

		// Include necessary files.
		$this->includes();
		// Load settings into the options panel.
		$this->register_settings();

		// Add a link to the settings panel on the plugin listing.
		add_filter( 'plugin_action_links_' . plugin_basename( $this->file ), array( $this, 'settings_link' ) );

		// Add front-end CSS.
		add_action( 'wp_enqueue_scripts', array( $this, 'stylesheet' ) );

		// Add TinyMCE button.
		add_action( 'admin_head', array( $this, 'add_tinymce_button' ) );

		// Add our shortcode.
		add_shortcode( 'scroll-box', array( $this, 'scroll_box' ) );

	}

	/**
	 * Sets up the main NG_Scroll_Box instance
	 *
	 * @access public
	 * @since  1.0.0
	 * @return NG_Scroll_Box
	 */
	public static function instance( $file = '', $version = '1.0.0' ) {
		if ( is_null( self::$_instance ) ) {
			self::$_instance = new self( $file, $version );
		}

		return self::$_instance;
	}

	/**
	 * Cloning is not allowed
	 *
	 * @access public
	 * @since  1.0.0
	 * @return void
	 */
	public function __clone() {
		// Cloning instances of the class is forbidden
		_doing_it_wrong( __FUNCTION__, __( 'Cheatin&#8217; huh?', $this->plugin_name ), $this->version );
	}

	/**
	 * Disable unserializing of the class
	 *
	 * @access public
	 * @since  1.0.0
	 * @return void
	 */
	public function __wakeup() {
		// Unserializing instances of the class is forbidden
		_doing_it_wrong( __FUNCTION__, __( 'Cheatin&#8217; huh?', $this->plugin_name ), $this->version );
	}

	/**
	 * Includes the required files.
	 *
	 * @access private
	 * @since  1.0.0
	 * @return void
	 */
	private function includes() {
		if ( ! class_exists( 'Nose_Graze_Settings' ) ) {
			require_once plugin_dir_path( $this->file ) . 'admin/ng-settings.class.php';
		}
	}

	/**
	 * Adds a link to the plugin's settings page on the listing.
	 *
	 * @param $links
	 *
	 * @access public
	 * @since  1.0.0
	 * @return array
	 */
	public function settings_link( $links ) {
		$settings_link = sprintf( '<a href="%s">' . __( 'Settings', $this->plugin_name ) . '</a>', admin_url( 'options-general.php?page=ng-scroll-box' ) );
		array_unshift( $links, $settings_link );

		return $links;
	}

	/**
	 * Registers the settings with the options panel.
	 *
	 * Updates the environment variable with the settings array.
	 *
	 * @uses  Nose_Graze_Settings
	 * @acess public
	 * @since 1.0.0
	 * @return void
	 */
	private function register_settings() {
		$fields = apply_filters( 'ng-scroll-box/settings', array(
			/* General */
			'general' => array(
				'title'  => __( 'General Settings', $this->plugin_name ),
				'desc'   => __( 'A few admin-related settings.', $this->plugin_name ),
				'fields' => array(
					'load_styles' => array(
						'id'   => 'load_styles',
						'name' => __( 'Load Default Styles', $this->plugin_name ),
						'desc' => __( 'If checked, a stylesheet will be loaded to set up the box and create the scrolling behaviour. Don\'t uncheck this unless you know what you\'re doing! If unchecked, the "Styling" tab will not actually work.', $this->plugin_name ),
						'type' => 'checkbox',
						'std'  => true
					),
					'add_tinymce' => array(
						'id'   => 'add_tinymce',
						'name' => __( 'Add TinyMCE Button', $this->plugin_name ),
						'desc' => __( 'If checked, a new button will be added to the visual editor to help you easily add boxes.', $this->plugin_name ),
						'type' => 'checkbox',
						'std'  => false
					),
				)
			),
			/* Styling */
			'styling' => array(
				'title'  => __( 'Styling', $this->plugin_name ),
				'desc'   => __( 'Customize the look of the box.', $this->plugin_name ),
				'fields' => array(
					'height'     => array(
						'id'   => 'height',
						'name' => __( 'Maximum Height', $this->plugin_name ),
						'desc' => __( 'The maximum height (in pixels) for the box. After this point, the box will start scrolling.', $this->plugin_name ),
						'type' => 'number',
						'std'  => '300'
					),
					'box_bg'     => array(
						'id'   => 'box_bg',
						'name' => __( 'Box Background Colour', $this->plugin_name ),
						'desc' => __( 'Choose a background colour for the box.', $this->plugin_name ),
						'type' => 'color',
						'std'  => '#f4f4f4'
					),
					'box_border' => array(
						'id'   => 'box_border',
						'name' => __( 'Box Border', $this->plugin_name ),
						'desc' => __( 'Configure the settings for the border around the box.', $this->plugin_name ),
						'type' => 'border',
						'std'  => array(
							'size'  => '2',
							'style' => 'solid',
							'color' => '#cccccc'
						)
					),
					'box_text'   => array(
						'id'   => 'box_text',
						'name' => __( 'Box Text Colour', $this->plugin_name ),
						'desc' => __( 'Choose a colour for the text inside the box.', $this->plugin_name ),
						'type' => 'color',
						'std'  => '#222222'
					),
				)
			)
		) );

		// Create a new settings object.
		$settings = new Nose_Graze_Settings(
			array(
				'page_title'   => __( 'NG Scroll Box', $this->plugin_name ),
				'menu_slug'    => $this->plugin_name,
				'options_slug' => 'ng_scroll_box',
			)
		);
		$settings->set_fields( $fields );

		// Update the environment variable with our settings array.
		$this->settings = $settings->get_options();
	}

	/**
	 * Public Stylesheet
	 *
	 * Adds the stylesheet to the front-end of the site and adds the
	 * custom settings "Styles" to that stylesheet.
	 *
	 * The stylesheet is not added if the "Load Default Styles" option
	 * is unchecked.
	 *
	 * @access public
	 * @since  1.0.0
	 * @return void
	 */
	public function stylesheet() {
		if ( ! array_key_exists( 'load_styles', $this->settings ) || empty( $this->settings['load_styles'] ) ) {
			return;
		}
		wp_enqueue_style( $this->plugin_name, plugins_url( 'css/front-end.css', dirname( __FILE__ ) ), array(), $this->version );
		wp_add_inline_style( $this->plugin_name, $this->inline_css() );
	}

	/**
	 * Inline CSS
	 *
	 * Gathers all customizations from the settings panel and puts them
	 * into CSS format.
	 *
	 * @access public
	 * @since  1.0.0
	 * @return string
	 */
	public function inline_css() {
		$css = '
		.scroll-box {
			background: ' . $this->settings['box_bg'] . ';
			border: ' . $this->settings['box_border']['size'] . 'px ' . $this->settings['box_border']['style'] . ' ' . $this->settings['box_border']['color'] . ';
			color: ' . $this->settings['box_text'] . ';
			height: ' . $this->settings['height'] . 'px;
		}
		';

		return $css;
	}

	/**
	 * Add TinyMCE Button
	 *
	 * Checks some permissions, etc. before adding filters to include
	 * our TinyMCE script and button.
	 *
	 * @access public
	 * @since  1.0.0
	 * @return void
	 */
	public function add_tinymce_button() {
		// If they opted not to include the TinyMCE button - bail.
		if ( ! array_key_exists( 'add_tinymce', $this->settings ) || empty( $this->settings['add_tinymce'] ) ) {
			return;
		}

		// If the user doesn't have the right permissions - bail.
		if ( ! current_user_can( 'edit_posts' ) && ! current_user_can( 'edit_pages' ) ) {
			return;
		}

		// Only add the buttons if the visual editor is enabled.
		if ( 'true' == get_user_option( 'rich_editing' ) ) {
			add_filter( 'mce_external_plugins', array( $this, 'add_tinymce_script' ) );
			add_filter( 'mce_buttons', array( $this, 'register_tinymce_button' ) );
		}
	}

	/**
	 * Add TinyMCE Script
	 *
	 * Adds our custom JavaScript file to TinyMCE. This is where the
	 * button functionality comes from.
	 *
	 * @param array $plugin_array
	 *
	 * @access public
	 * @since  1.0.0
	 * @return array
	 */
	public function add_tinymce_script( $plugin_array ) {
		$plugin_array['scroll_box_button'] = plugins_url( 'admin/assets/js/tinymce.js', dirname( __FILE__ ) );

		return $plugin_array;
	}

	/**
	 * Register TinyMCE Button
	 *
	 * @param array $buttons
	 *
	 * @access public
	 * @since  1.0.0
	 * @return array
	 */
	public function register_tinymce_button( $buttons ) {
		array_push( $buttons, 'scroll_box_button' );

		return $buttons;
	}

	/**
	 * Scroll Box Shortcode
	 *
	 * @param array  $atts    Shortcode attributes
	 * @param string $content Text inside the shortcode
	 *
	 * @access public
	 * @since  1.0.0
	 * @return string
	 */
	public function scroll_box( $atts, $content = '' ) {
		return '<div class="scroll-box">' . do_shortcode( $content ) . '</div>';
	}

}